//*************************************************************************************************
//
//	Description:
//		HDRMethods_Combine.fx - Final Pass of the HDR
//
//	<P> Copyright (c) 2007 Blimey! Games Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     01/11/2007  0.1           Created
//		TNettleship     01/22/2007  0.2           Fixed bilinear problems with fp16 targets on NV7x
//																							series.
//		BIrvine			    05/24/2007  0.3           Clamped bottom end of luminance calculation result
//																							to avoid issues with negative light.
//		TNettleship			05/29/2007	0.4						Added support for viewports.
//		TNettleship			05/29/2007	0.5						Added support for gradual exposure adaptation.
//		BIrvine					06/08/2007	0.6						Added temporary code to clamp high luminance values.
//		TMann						05/07/2007	0.7						Added PS3 support and edge detection
//		TNettleship			06/26/2007	0.8						Improved the pixel luminance calc & added clamping
//																							to the range of the tone mapping effect.
//		TNettleship     07/24/2007  0.9           Made sure samplers aren't using anisotropic filtering.
//	<TABLE>
//
//*************************************************************************************************

#include "stddefs.fxh"


//------------------------------------------------------------------
//
//  GLOBAL VARIABLES
//

//------------------------------------------------------------------
//
// Viewport into source framebuffer
//

float2 viewportOrigin;
float2 viewportScale;

#if defined(_PS3_)
#define _MINFILTER	LinearMipMapLinear
#else
#define _MINFILTER	Linear
#endif

//------------------------------------------------------------------
//
// Luminance calculation
// 

texture luminanceInputTexture : TEXTURE;
sampler luminanceInput : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < luminanceInputTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	SET_NO_ANISOTROPY
};


//------------------------------------------------------------------
//
// Bloom calculation
//
float bloomFactor;											// Used in the post-processing, but also useful here


texture bloomInputTexture : TEXTURE;
sampler bloomInput : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < bloomInputTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
#ifdef _XBOX
	MinFilter = Linear;
	MagFilter = Linear;
#else
	MinFilter = Point;
	MagFilter = Point;
#endif
	MipFilter = None;
	SET_NO_ANISOTROPY
};


//------------------------------------------------------------------
//
// Final pass
//

float adaptationRate;															// Rate of adaptation to exposure change
float bloomTextureWidth;													// The width of the texture in 'bloom'
float bloomTextureHeight;													// The height of the texture in 'bloom'
float ooBloomTextureWidth;												// The reciprocal width of the texture in 'bloom'
float ooBloomTextureHeight;												// The reciprocal height of the texture in 'bloom'

texture HDRSceneTexture : TEXTURE;								// The HDR rendered framebuffer
sampler HDRSceneInput : SAMPLER = sampler_state
{
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	Texture = < HDRSceneTexture >;
	AddressU  = Clamp;
	AddressV  = Clamp;
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = None;
	SET_NO_ANISOTROPY
};

//------------------------------------------------------------------
//
// Vertex shader
//

struct VSINPUT
{
	float3 position : POSITION;								// Texture coords as a vertex position
	float2 texCoord : TEXCOORD0;
};

struct VSOUTPUT
{
	float4 position : POSITION;
	float2 texCoord : TEXCOORD0;
};



VSOUTPUT HDRVertexShader( VSINPUT _input )
{
	VSOUTPUT output;

	// Just pass through the texture values to the pixel shader
	output.position = float4( _input.position.xyz, 1.0f );
	output.texCoord = _input.texCoord;

	return output;
}



//
// Pixel shaders
//

struct PSINPUT
{
	float2 texCoord : TEXCOORD0;
};

struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE colour : COLOR0;
};

//------------------------------------------------------------------
//
// Function calculates the luminance of a HDR pixel's colour.
//
//------------------------------------------------------------------

float CalculatePixelLuminance( float4 _colour )
{
	const float3 weights = { 0.29f, 0.50f, 0.27f };
	float lLuminance = dot( _colour.rgb, weights );

	return lLuminance;
}

// Params
float4 autoValues0;
float4 autoValues1;
float _whitePoint;
float _key;
float lowerClamp;
float upperClamp;

//------------------------------------------------------------------
//  SHADER ENTRY POINT
//------------------------------------------------------------------
PSOUTPUT HDR_CombinePass_FragmentShader( PSINPUT _input )
{
	// Read the HDR value that was computed as part of the original scene
	float4 c = tex2D( HDRSceneInput, _input.texCoord );

	// Clamp colours to PS3 range
//	c.r=min(c.r, 16.0f);
//	c.g=min(c.g, 16.0f);
//	c.b=min(c.b, 16.0f);

	// Read the luminance value, target the centre of the texture
	// which will map to the only pixel in it!
	float4 l = tex2D( luminanceInput, float2( 0.5f, 0.5f ) );

	// Apply the the viewport xform to the input texcoord, to get the bloom texcoord
	float2 bloomCoord = ( _input.texCoord - viewportOrigin ) / viewportScale;

#ifdef _XBOX

	// Xbox is using a format that can be sampled with linear filtering

	float4 b;
	if ( bloomFactor > 0.0f )
	{
		b = tex2D( bloomInput, bloomCoord );        
	}
	else
	{
		b = float4( 0.0f, 0.0f, 0.0f, 0.0f );
	}

#else

	float4 b;
	if ( bloomFactor > 0.0f )
	{
		// Compute the blur value using a bilinear filter
		// It is worth noting that if the hardware supports linear filtering of a
		// floating point render target that this step can probably be skipped.
		float xWeight = frac( bloomCoord.x * bloomTextureWidth ) - 0.5;
		float xDir = xWeight;
		xWeight = abs( xWeight );
		xDir /= xWeight;
		xDir *= ooBloomTextureWidth;

		float yWeight = frac( bloomCoord.y * bloomTextureHeight ) - 0.5;
		float yDir = yWeight;
		yWeight = abs( yWeight );
		yDir /= yWeight;
		yDir *= ooBloomTextureHeight;

		// sample the blur texture for the 4 relevant pixels, weighted accordingly
		b = ( ( 1.0f - xWeight) * (1.0f - yWeight) )	* tex2D( bloomInput, bloomCoord );        
		b +=			 ( xWeight * ( 1.0f - yWeight ) )					* tex2D( bloomInput, bloomCoord + float2( xDir, 0.0f ) );
		b +=			 ( yWeight * ( 1.0f - xWeight ) )					* tex2D( bloomInput, bloomCoord + float2( 0.0f, yDir ) );
		b +=			 ( xWeight * yWeight )										* tex2D( bloomInput, bloomCoord + float2( xDir, yDir ) );
	}
	else
	{
	 b = float4( 0.0f, 0.0f, 0.0f, 0.0f );
	}
	
#endif

	// Compute the actual colour:

	// Reinhard's tone mapping equation (See Eqn#3 from 
	// "Photographic Tone Reproduction for Digital Images" for more details) is:
	//
	//      (      (   Lp    ))
	// Lp * (1.0f +(---------))
	//      (      ((Lm * Lm)))
	// -------------------------
	//         1.0f + Lp
	//
	// Lp is the luminance at the given point, this is computed using Eqn#2 from the above paper:
	//
	//        exposure
	//   Lp = -------- * HDRPixelIntensity
	//          l.r
	//
	// The exposure ("key" in the above paper) can be used to adjust the overall "balance" of 
	// the image. "l.r" is the average luminance across the scene, computed via the luminance
	// downsampling process. 'HDRPixelIntensity' is the measured brightness of the current pixel
	// being processed.

	// Work out min/max
	float scale=(2.7f-1.0f)/16.0f;
	float lGoal = (exp(l.r)-1.0f)/scale;
	float key;
	float white;
	
	lGoal=clamp(lGoal, lowerClamp, upperClamp);
	
	key=_key;
	white=_whitePoint;
	
	float lw2=white;
	lw2=lw2*lw2;
	float lum=CalculatePixelLuminance( c );
	lum=(key*lum)/lGoal;
	float toneScalar=(lum*(1.0f+lum/lw2))/(1.0f+lum);
	
	float scale_factor = 1.0f / lGoal;
	toneScalar  *= scale_factor * key;
	
	c = ( c * toneScalar ) + b;

	// Return the fully composed colour
	PSOUTPUT output;
	output.colour = float4( c.rgb, 1.0f );
	return output;
}


technique HDR_CombinePass
{
	pass Pass0
	{
		ZEnable = 0;
		ZWriteEnable = false;
		AlphaBlendEnable = false;
		AlphaTestEnable = false;
#ifdef _PS3_
		VertexShader = compile sce_vp_rsx HDRVertexShader();
		PixelShader = compile sce_fp_rsx HDR_CombinePass_FragmentShader();
#else		
		VertexShader = compile vs_3_0 HDRVertexShader();
		PixelShader = compile ps_3_0 HDR_CombinePass_FragmentShader();
//		PixelShader = compile ps_3_0 TexOutput_FragmentShader();
#endif
	}
}
